<?php
/*
Plugin Name: Link Indication
Plugin URI: http://sw-guide.de/wordpress/plugins/link-indication/
Description: Adds CSS class attributes to external links and optionally specific attributes to any other link types such as wikipedia.org, flickr, imdb, file extensions like .pdf or .zip, etc. Thereby you can indicate your links, e.g. by images, for characterizing your types of links. Furthermore you can add target="blank" to all external links, rel="nofollow" to specific hyperlinks or display websnapr preview thumbnails. Navigate to <a href="admin.php?page=link-indication.php">Settings &rarr; Link Indication</a>.
Version: 4.3
Author: Michael Wöhrer
Author URI: http://sw-guide.de/

    ----------------------------------------------------------------------------
   	      ____________________________________________________
         |                                                    |
         |         Link Indication WordPress Plugin           |
         |____________________________________________________|

	                  Copyright © Michael Wöhrer 
	                    <http://sw-guide.de>
                (michael dot woehrer at gmail dot com)

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License <http://www.gnu.org/licenses/> for 
	more details.

	----------------------------------------------------------------------------
*/

require_once ( dirname(__FILE__) . '/inc.swg-plugin-framework.php');

class LinkIndication extends LinkIndication_SWGPluginFramework {

	/**
	 * For applying the link indication
	 */	 	
	function ApplyLinkIndication() {

		switch ($this->g_opt['mwli_apply']) {
			case '1':	// Apply to content & comments
				add_filter('the_content', array(&$this, 'LinkIndicationMain'), 10);
				add_filter('the_excerpt', array(&$this, 'LinkIndicationMain'), 10);				
				add_filter('comment_text', array(&$this, 'LinkIndicationMain'), 10);
				break;
			case '2':	// Apply to content only
				add_filter('the_content', array(&$this, 'LinkIndicationMain'), 10);
				add_filter('the_excerpt', array(&$this, 'LinkIndicationMain'), 10);				
				break;
			case '3':	// Apply to entire blog but not to WP admin
				if ( strpos($_SERVER['REQUEST_URI'], 'wp-admin') === false ) {
					// Calling ob_start here causes that the plugin is not applied if gzip compression is turned on in the options.
					// wp-config.php calls wp-settings.php, which loads the plugins. gzip_compression() isn't called until after that, 
					// in wp-blog-header.php. But since we want to apply after the compression is turned on, we just need to hook the 
					// earliest thing after that we can -- which seems to be the template_redirect hook.
					add_action('template_redirect', array(&$this, 'ForAddAction_template_redirect_ob_start'));
				}
				break;
		} // switch

		if ($this->g_opt['mwli_snapr'] == '1') {
			add_action('wp_head', array(&$this, 'ForAddAction_wp_head_snappr'));
		}
		if ($this->g_opt['mwli_usecssfile'] == '1') {
			add_action('wp_head', array(&$this, 'ForAddAction_wp_head_style'));
		}
	}
	
	function ForAddAction_template_redirect_ob_start() {
		ob_start(array(&$this, 'LinkIndicationMain'));
	}
	function ForAddAction_wp_head_snappr() {
		// Apply websnapr
		echo "\n\t".'<!-- Generated by Link Indication Plugin ' . $this->g_info['Version'] . ' - http://sw-guide.de/ -->'
			. "\n\t".'<script type="text/javascript">var bubbleImagePath="' . $this->GetPluginURL() . 'images/snapr-bg.png"</script>'
			. "\n\t".'<script type="text/javascript" src="' . $this->GetPluginURL() . 'js/snapr_previewbubble.js"></script>'
			. "\n\t".'<!-- /Link Indication Plugin -->'
			. "\n\t";
	}
	function ForAddAction_wp_head_style() {
		// Apply link-indication_style.css
		echo "\n\t".'<!-- Generated by Link Indication Plugin ' . $this->g_info['Version'] . ' - http://sw-guide.de/ -->'
			. "\n\t".'<style type="text/css" media="screen">'
			. "\n\t\t".'@import url( '. $this->GetPluginURL() .'link-indication_style.css );'
			. "\n\t".'</style>'
			. "\n\t".'<!-- /Link Indication Plugin -->'
			. "\n\t";
	}

	/**
	 * The main function
	 */	 	
	function LinkIndicationMain($content) {
			/* ***************** Pattern ******************
			match1 (.*?):
				everything between <a and href
			match2 (href=("|\')(.*?)("|\')(.*?)|):
				the entire href="url", just to support anchors with missing href as well
			match3 ("|\'):
				double or single quoted link. We use it instead of ["\'] so that
				also non quoted URLs are possible
			match4 (.*?):
				the URL
			match5 ("|\'):
				double or single quoted link
			match6 (.*?):
				everything between the href="..." and ">"
			match7 (.*?):
				text of the link
		**********************************************/
		$pattern = '/<a (.*?)(href=("|\')(.*?)("|\')(.*?)|)>(.*?)<\/a>/i'; // don't remove space between '<a' and '(.*?)' to avoid that plugin is not applied when abbr or acronyms are used 
		$result = preg_replace_callback($pattern,array(&$this,'ParseLinks'),$content);
	
		return $result;
	}

	/**
	 * Parses the links
	 */
	function ParseLinks($matches){

		/**********************
		* If there is no href...
		*********************/
		if ($matches[4] == '') {
			// There is no href so we return it just at is was provided;
			return '<a ' . $matches[1] . $matches[2] . '>' . $matches[7] . '</a>';	// Don't remove space in '<a ' or it will cause an error for the "more" anchor: <aid="more-123"> 
		}

		/**********************
		* Because of problems with JS links we skip these
		*********************/
		if (strpos(strtolower($matches[4]), 'javascript') !== false) {
			return '<a ' . $matches[1] . $matches[2] . '>' . $matches[7] . '</a>';
		}
	
		/**********************
		* Prepare blog urls
		*********************/	
		$blogurlsArray = explode(' ', $this->g_opt['mwli_blogurls']);
	 	$loopcount = 0;
		foreach ($blogurlsArray as $loopval) {
			$blogurlsArrayClean[$loopcount] = $this->GetDomainnameFromUri($loopval);
			$loopcount++;
		}
	
		/**********************
		* Get the domain name
		*********************/
		// Since plugin version 4.2, we do no longer use {$domainname_link = $this->GetDomainnameFromUri($matches[4]);}
		// We don't want to use the domain name only but the entire URL for the search - but without its parameters...
		$domainname_link = $matches[4];
		if ( strpos($domainname_link, '?')  !== false) {
			$domainname_link = substr($domainname_link, 0, strpos($domainname_link, '?'));
		}

		/**********************
		* Get the type of the link
		*********************/
		$linktype = 'internal';	// default is internal, e.g. <a href="index.php?option=com...
		if (substr($matches[4], 0, 4) == 'http') $linktype = 'external-http';	// https as well...
		if (substr($matches[4], 0, 3) == 'ftp') $linktype = 'external-ftp';
		if (substr($matches[4], 0, 7) == 'mailto:') $linktype = 'mailto';
	
	
		if ( ($linktype == 'external-http') and ( in_array($this->GetDomainnameFromUri($matches[4]), $blogurlsArrayClean) ) ) {
			// -- the link is internal but with leading http...
			$linktype = 'internal';
		}
	
		/**********************
		* Check if the link is user defined or if a file extension needs to be applied
		*********************/
		$loopCount = 0;
		foreach ($this->g_opt['mwli_searchstrings'] as $loopVar) {
			if ( ($loopVar != '') ) {
				// User can enter more than one value and separate them by pipe symbol '|' or comma or semicolon. 
				// So we need an arrray of all values and check each by looping through the array.
				// By the way, explode returns an array containing the source string 
				// if the '|' is not contained in the string.
				$loopVar = str_replace(';', '|', $loopVar);
				$loopVar = str_replace(',', '|', $loopVar);
				$searchstringArray = explode('|', $loopVar);
				
				foreach ($searchstringArray as $loopSearchString) {
					$loopSearchString = str_replace(' ', '', $loopSearchString);	// remove all spaces
					if ( $loopSearchString != '' ) {
						// Finally we have a matching
						if ( ($linktype == 'external-http') && ($this->g_opt['mwli_types'][$loopCount] == 'External URL') && (strpos($domainname_link, $loopSearchString) !== false) ) {
							// external URL... 
							$linktype = 'external-userdefined';
							$csssetting = $this->g_opt['mwli_cssclasses'][$loopCount];
							break;							
						} elseif ((substr($matches[4], -(strlen($loopSearchString) + 1)) === ('.' . $loopSearchString))
						          && ($this->g_opt['mwli_types'][$loopCount] == 'File Extension') ) {
							$linktype = 'filename-extension';
							$csssetting = $this->g_opt['mwli_cssclasses'][$loopCount];
							break;
						}
					}
				}
			}
			if ( $csssetting != '' ) {
				// exit loop now, no need to continue...
				break;
			}				
			$loopCount++;
		}

	
		/**********************
		* For external links, we apply target="_blank" (if there is not already a target attribute in the anchor tag)
		*********************/
		$targetblank = '';
		if ( (substr($linktype, 0, 8) == 'external') && ($this->g_opt['mwli_targetblank'] == '1') ) {
				if ( (strpos($matches[1] . $matches[6], 'target=') === false) ) {
					// there is no target=, so we add target="_blank"
					$targetblank = 'target="_blank"';
				}
		}
	
		/**********************
		* For some links, we apply rel="nofollow" if activated in option and is not already there...
		*********************/
		$nofollow = '';
		$nofollowWebsiteArray = explode(' ', $this->g_opt['mwli_nofollow']);
		foreach ($nofollowWebsiteArray as $loopVar) {
			if (strpos($matches[4], $loopVar) !== false) {
				if ( ! preg_match("/rel=[\"\'].*?nofollow.*?[\"\']/i", $matches[1] . $matches[6] ) ) {
					$nofollow = 'rel="nofollow"';
				}				
			}
		}

		
		/**********************
		* Build the css class
		*********************/
		switch ($linktype) {
			case 'external-http':
				$csssetting = $this->g_opt['mwli_external'];
				break;
			case 'external-userdefined':
				// nothing to do since we have set the $csssetting above.
				break;
			case 'filename-extension':
				// nothing to do since we have set the $csssetting above.
				break;
			case 'external-ftp':
				$csssetting = $this->g_opt['mwli_ftp'];
				break;
			case 'mailto':
				$csssetting = $this->g_opt['mwli_mailto'];
				break;
			default: // internal
				$csssetting = $this->g_opt['mwli_internal'];
		} //switch

	  	/**********************
	  	* If it is an image url, do not add CSS, but add image link class
	  	*********************/
	  	if (substr_count($matches[7] , '<img ') > 0) {
	    	if ($this->g_opt['mwli_toimages'] != '1') {
	  			$csssetting = '';
	  		}
	    	if ($this->g_opt['mwli_imagelinks'] !== '') {
	    	  if ($csssetting === '') {
	      	  // nothing has been set or it has been reset above
	      	  $csssetting = $this->g_opt['mwli_imagelinks'];
	    	  } else {
	    	    // append the image link class
	    	    $csssetting = $csssetting . ' ' . $this->g_opt['mwli_imagelinks'];
	    	  }
	    	}
	  	}


		/**********************
		* Do not add CSS if we deactivated it;
		*********************/
		if ($this->g_opt['mwli_applycssclassattr'] != '1') {
			$csssetting = '';
		}

		/**********************
		* Add websnapr class to all external links...
		*********************/
		if (($this->g_opt['mwli_snapr'] == '1') && (substr($linktype, 0, 8) == 'external') ) {
			$csssetting = $csssetting . ($csssetting == '' ? '' : ' ') . 'previewlink';
		}

		/**********************
		* Do not apply the class if there is already a class inside the anchor tag
		*********************/
		if ( (strpos($matches[1] . $matches[6], 'class=') !== false) ) {
			$csssetting = '';
		}

		// Final class
		if ($csssetting == '') {
			$finalclass = '';
		} else {
			$finalclass = 'class="' . $csssetting . '"';
		}

		/**********************
		* Final preparation
		*********************/	
		// put together the result
		$final_stuff = $matches[1] . ' ' . $matches[6] . ' ' . $targetblank . ' ' . $nofollow . ' ' . $finalclass;
		// replace multiple white spaces with a single space
		$final_stuff = eregi_replace('[[:space:]]+', ' ', $final_stuff);
		// remove leading and trailing white space
		$final_stuff = trim($final_stuff);
		// replace single quotes (') with double quotes (")
		//$final_stuff = str_replace('\'', '"', $final_stuff);	// As of version 4.2: We do no longer do this due to problems with JS links

		/**********************
		* Return the result
		*********************/
		return '<a href="' . $matches[4] . '" ' . $final_stuff . '>' . $matches[7] . '</a>';
	
	} // function ParseLinks


	/**
	 * Retrieve the domain name from the URI. We consider sub domains as well.
	 */	 
	function GetDomainnameFromUri($uri){
	    $exp1 = '/^(http|https|ftp)?(:\/\/)?(www.)?([^\/]+)/i';
		preg_match($exp1, $uri, $matches);
		if (isset($matches[4])) {
	    	return $matches[4];
	    } else {
			return '';
		}
	}


	/**
	 * Convert option prior to save ("COPTSave"). 
	 * !!!! This function is used by the framework class !!!!
	 */
	function COPTSave($optname) {
		if ( ($optname == 'mwli_searchstrings') || ($optname == 'mwli_cssclasses') || ($optname == 'mwli_types') ) {
			// Special treatment for 3 fields
			$urlPostArray = $_POST['mwli_searchstrings'];
			$cssPostArray = $_POST['mwli_cssclasses'];
			$typePostArray = $_POST['mwli_types'];
			$loopCount = 0;
			if (is_array($urlPostArray)) {
				foreach ($urlPostArray as $loopVarURL) {
					$loopVarURL = str_replace(' ', '', $loopVarURL);	// Strip whitespace
					$loopVarCSS = str_replace(' ', '', $cssPostArray[$loopCount]);	// Get css and strip whitespace
					$loopVarType = $typePostArray[$loopCount]; // Get type
					if ( $loopVarURL != '' ) {
						$urlResultArray[] = $loopVarURL;
						$cssResultArray[] =  $loopVarCSS;
						$typeResultArray[] =  $loopVarType;
					}
					$loopCount++;
				}
			}
		}
		switch ($optname) {
			case 'mwli_blogurls':
			    return $this->LineBreakToWhiteSpaceAndFormatURL($_POST[$optname]);
			case 'mwli_nofollow':
				return $this->LinebreakToWhitespace($_POST[$optname]);
			case 'mwli_searchstrings':
				return $urlResultArray;
			case 'mwli_cssclasses':
				return $cssResultArray;
			case 'mwli_types':
				return $typeResultArray;
			default:
				return $_POST[$optname];
		} // switch
	}

	/**
	 * Convert option before HTML output ("COPTHTML"). 
	 * *NOT* used by the framework class
	 */
	function COPTHTML($optname) {
		$optval = $this->g_opt[$optname];
		switch ($optname) {
			case 'mwli_nofollow':
				return $this->WhitespaceToLinebreak($optval);
			case 'mwli_blogurls':
				return $this->WhitespaceToLinebreak($optval);
			default:
				return $optval;
		} // switch
	}

	/**
	 * Converts line breaks to whitespace and formats the URLs. 
	 * Applied before storing value in database
	 */
	function LineBreakToWhiteSpaceAndFormatURL($input) {
	
		// Remove white spaces
		$input = str_replace(' ', '', $input);
	
		// Replace linebreaks with white space, considering both \n and \r
		$input = preg_replace("/\r|\n/s", ' ', $input);
	
		// Create result. We create an array and loop thru it but do not consider empty values. 
		$sourceArray = explode(' ', $input);
		$loopcount = 0;
		$result = '';
		foreach ($sourceArray as $loopval) {
	
			if ($loopval <> '') {
	
				// Remove everything after the tld
				$exp1 = '/^(http:\/\/|https:\/\/|ftp:\/\/)?([^\/]+)/i';
				preg_match($exp1, $loopval, $matches);
				if ($matches[1] == '') $matches[1] = 'http://';	// Add http:// if there is nothing...
				$loopval = $matches[1] . $matches[2];  
				
				// Create separator
				$sep = '';
				if ($loopcount >= 1) $sep = ' ';
				
				// result
				$result .= $sep . $loopval;
			
				$loopcount++;				
			}
		}
		return $result;
	
	}

	/**
	 * Plugin Options
	 */
	function PluginOptionsPage() {

		//Add options
		$this->AddContentMain(__('style.css and link-indication_style.css',$this->g_info['ShortName']), 
			'<p>'
			. __('We use',$this->g_info['ShortName'])
			. ' <a href="http://codex.wordpress.org/Glossary#CSS">CSS</a> '
			. __('for the indication of links. You can either add appropriate CSS rules to your <code>style.css</code> of your themes directory',$this->g_info['ShortName'])
			. ' (<code>/wp-content/themes/&hellip;</code>)'
			. __('or you apply the file',$this->g_info['ShortName'])
			. ' <code>link-indication_style.css</code> '
			. __('in the plugin\'s directory by activating the following option. In this case the images in',$this->g_info['ShortName']).' <code>' . $this->GetPluginURL() . 'images/</code> '
			. __('will be used in the css file.',$this->g_info['ShortName']).'<br />'
			. __('Please don\'t forget to refresh the browser\'s cache (e.g. CTRL + F5 in Firefox) after you have modified the CSS file, otherwise you will not see your changes immediately.',$this->g_info['ShortName'])
			.'			
			</p>		
			<ul>
				<li>
					<input name="mwli_usecssfile" type="checkbox" id="mwli_usecssfile" value="1" '.($this->COPTHTML('mwli_usecssfile')=='1'?'checked="checked"':'').' /> 
					<label for="mwli_usecssfile">'.__('Apply <code>link-indication_style.css</code> from the plugin\'s directory',$this->g_info['ShortName']).'</label>
				</li>
			</ul>'
		);

		//Add options
		$this->AddContentMain(__('Plugin\'s Application',$this->g_info['ShortName']), '
			<p style="line-height: 2em;">
				<input id="radioa1" type="radio" name="mwli_apply" value="1" ' . ($this->COPTHTML('mwli_apply')=='1'?'checked="checked"':'') . ' />
				<label for="radioa1">'.__('Apply plugin to content and comments [default]',$this->g_info['ShortName']).'</label>
				<br />
				<input id="radioa2" type="radio" name="mwli_apply" value="2" ' . ($this->COPTHTML('mwli_apply')=='2'?'checked="checked"':'') . ' />
				<label for="radioa2">'.__('Apply plugin to content only and not to comments',$this->g_info['ShortName']).'</label>
				<br />	
				<input id="radioa3" type="radio" name="mwli_apply" value="3" ' . ($this->COPTHTML('mwli_apply')=='3'?'checked="checked"':'') . ' />
				<label for="radioa3">'.__('Apply plugin to the entire blog incl. header, sidebar, footer, etc.',$this->g_info['ShortName']).'</label>
			</p>				
		');

		//Add options
		$this->AddContentMain(__('Apply indicators and attributes to links',$this->g_info['ShortName']), '
		<h4>'.__('Apply CSS Class Attributes',$this->g_info['ShortName']).'</h4>
		<p>'.__('This is the main purpose of the plugin: applying CSS class attributes to links. Further options are below in the section &laquo;CSS Class Attributes&raquo;.',$this->g_info['ShortName']).'
		<ul>
			<li>
				<input name="mwli_applycssclassattr" type="checkbox" id="mwli_applycssclassattr" value="1" '.($this->COPTHTML('mwli_applycssclassattr')=='1'?'checked="checked"':'').' /> 
				<label for="mwli_applycssclassattr">'.__('Apply CSS Class Attributes [default: yes]',$this->g_info['ShortName']).'</label>
			</li>
		</ul>
		<h4>'.__('target="_blank"',$this->g_info['ShortName']).'</h4><p>'
		.__('This option will add',$this->g_info['ShortName'])
		.' <a href="http://www.google.com/search?hl=en&q=target%3D%22_blank%22">target="_blank"</a> '
		.__('to all external hyperlinks. Please note that target="_blank" is not recommended according to',$this->g_info['ShortName']) 
		.' <a href="http://www.w3.org/">W3C</a> '
		.__('and your website will not validate as XHTML Strict when activating this option.',$this->g_info['ShortName']).'

		<ul>
			<li>
				<input name="mwli_targetblank" type="checkbox" id="mwli_targetblank" value="1" '.($this->COPTHTML('mwli_targetblank')=='1'?'checked="checked"':'').' /> 
				<label for="mwli_targetblank">'.__('Add <em>target="_blank"</em> to all external hyperlinks',$this->g_info['ShortName']).'</label>
			</li>
		</ul>
		<h4>'.__('Websnapr preview thumbnails',$this->g_info['ShortName']).'</h4>
			<p>'.__('Displays overlay bubbles showing hyperlink target thumbnails, uses',$this->g_info['ShortName']).' <a href="http://www.websnapr.com/previewbubble/">websnapr.com</a> JS'
				.__('. Please note that visitors may find these thumbnail previews pretty annoying.',$this->g_info['ShortName']).'</p>
		<ul>
			<li>
				<input name="mwli_snapr" type="checkbox" id="mwli_snapr" value="1" '.($this->COPTHTML('mwli_snapr')=='1'?'checked="checked"':'').' /> 
				<label for="mwli_snapr">'.__('Display websnapr preview thumbnails for all external links',$this->g_info['ShortName']).'</label>
			</li>
		</ul>
		<h4>'.__('Add \'nofollow\' attribute',$this->g_info['ShortName']).'</h4>
			<p>' . __('When an URL contains any of these expressions, a <em><a href="http://en.wikipedia.org/wiki/Nofollow">rel="nofollow"</a></em> attribute will be assigned. One expression per line. It will match inside URLS, so the expression <code>wp-admin</code> will match <code>http://testblog.com/<strong>wp-admin</strong>/options-general.php</code>.',$this->g_info['ShortName']).'<br /><small>'.__('Example: If you want to assign nofollow to all links to Wikipedia, then enter <em>wikipedia.org</em>. This will apply nofollow to all links including subdomains like <em>de.wikipedia.org</em>, <em>fr.wikipedia.org</em>, etc. By the way, <a href="http://en.wikipedia.org/wiki/Wikipedia:Wikipedia_Signpost/2007-01-22/Nofollow">Wikipedia is adding nofollow tags to all external links</a>, therefore it is listed below as default value.',$this->g_info['ShortName']).'</small>'
			. '<textarea name="mwli_nofollow" id="mwli_nofollow" cols="100%" rows="2" >' . $this->COPTHTML('mwli_nofollow') . '</textarea>'
		);


		// Add options; for the link types
		# Get options
		$tmpSearchStringArray = $this->COPTHTML('mwli_searchstrings');
		$tmpCssArray = $this->COPTHTML('mwli_cssclasses');
		$tmpTypeArray = $this->COPTHTML('mwli_types');
		# Add a few empty values
		$tmpSearchStringArray[] = ''; $tmpCssArray[] = '';
		$tmpSearchStringArray[] = ''; $tmpCssArray[] = '';
		$tmpSearchStringArray[] = ''; $tmpCssArray[] = '';
		$tmpSearchStringArray[] = ''; $tmpCssArray[] = '';
		$tmpSearchStringArray[] = ''; $tmpCssArray[] = '';
		// Loop thru each value...
		$loopCount = 0;
		$fieldoutput = '';
		foreach($tmpSearchStringArray as $loopVar) {
			$fieldoutput .= '
			<tr valign="center"> 
				<th  align=right scope="row">' . ($loopCount + 1) . '.&nbsp;&nbsp;</th>
				<td><select name="mwli_types[]" id="mwli_types[]">
					<option style="width:110px" value="External URL"' . ($tmpTypeArray[$loopCount]=='External URL'?' selected':'') . '>'.__('External URL',$this->g_info['ShortName']).'</option>
					<option value="File Extension"' . ($tmpTypeArray[$loopCount]=='File Extension'?' selected':'') . '>'.__('File Extension',$this->g_info['ShortName']).'</option>
				</select></td>
				<td><input name="mwli_searchstrings[]" type="text" id="mwli_searchstrings[]" value="' . $loopVar . '" size="20" /></td>
				<td><input name="mwli_cssclasses[]" type="text" id="mwli_cssclasses[]" value="' . $tmpCssArray[$loopCount] . '" size="15" /></td>
				
			</tr>
			';
			$loopCount++;
		}
		$this->AddContentMain(__('CSS Class Attributes',$this->g_info['ShortName']),'
			<p>'.__('These CSS class attributes will be added to your hyperlinks. Please make sure that you have activated the option &laquo;Apply CSS Class Attributes&raquo; above.',$this->g_info['ShortName']).'
			<br />'.__('Example: If you have entered "liexternal" in the field "External links:", external links like <code>&lt;a href="http://www.google.com"&gt;Google&lt;/a&gt;</code> will become to <code>&lt;a <strong>class="liexternal"</strong> href="http://www.google.com"&gt;Google&lt;/a&gt;</code>.',$this->g_info['ShortName']).'
			</p>
			<table width="100%" cellspacing="2" cellpadding="5"> 
			<tr valign="center"> 
			<th align=left width="140px" scope="row"><label for="mwli_external">'.__('External links:',$this->g_info['ShortName']).'</label></th> 
			<td width="30px"><input name="mwli_external" type="text" id="mwli_external" value="' . $this->COPTHTML('mwli_external') . '" size="15" /></td> 
			<td><small>'.__('General links to external websites such as google.com, wordpress.org etc.',$this->g_info['ShortName']).'</small></td>
			</tr> 
			<tr valign="center"> 
				<th align=left scope="row"><label for="mwli_ftp">'.__('FTP links:',$this->g_info['ShortName']).'</label></th> 
				<td><input name="mwli_ftp" type="text" id="mwli_ftp" value="' . $this->COPTHTML('mwli_ftp') . '" size="15" /></td> 
				<td><small>'.__('Links to FTP server (ftp://...).',$this->g_info['ShortName']).'</small></td>
			</tr>
			<tr valign="center"> 
				<th align=left scope="row"><label for="mwli_mailto">'.__('mailto links:',$this->g_info['ShortName']).'</label></th> 
				<td><input name="mwli_mailto" type="text" id="mwli_mailto" value="' . $this->COPTHTML('mwli_mailto') . '" size="15" /></td> 
				<td><small>'.__('E-mail links, e.g. mailto:test@site.com.',$this->g_info['ShortName']).'</small></td>
			</tr>
			<tr valign="center"> 
				<th align=left scope="row"><label for="mwli_internal">'.__('Internal links:',$this->g_info['ShortName']).'</label></th> 
				<td><input name="mwli_internal" type="text" id="mwli_internal" value="' . $this->COPTHTML('mwli_internal') . '" size="15" /></td> 
				<td><small>'.__('All internal links.',$this->g_info['ShortName']).'</small></td>
			</tr>

	  		<tr valign="center">
	  			<th align=left scope="row"><label for="mwli_imagelinks">'.__('Image links:',$this->g_info['ShortName']).'</label></th> 
	  			<td><input name="mwli_imagelinks" type="text" id="mwli_imagelinks" value="' . $this->COPTHTML('mwli_imagelinks') . '" size="15" /></td> 
				<td><small>'.__('All links containing an image (<em>&lt;img&gt;</em> tags inside the <em>&lt;a&gt;</em> tag).',$this->g_info['ShortName']).'</small></td>
	  		</tr>

			</table>

			<hr />
			<h4>'.__('Specific link types',$this->g_info['ShortName']).'</h4>
			<table width="100%" cellspacing="2" cellpadding="5" class="editform">
				<tr valign="center"> 
					<th width="50px" scope="col">&nbsp;</th>
					<th style="text-align: left;" width="150px" scope="col">'.__('Type',$this->g_info['ShortName']).'</th>
					<th style="text-align: left;" width="180px" scope="col">'.__('Search string',$this->g_info['ShortName']).'</th>
					<th style="text-align: left;" scope="col">'.__('CSS class',$this->g_info['ShortName']).'</th>
				</tr>'
			. $fieldoutput
			. '</table>
			<h4 style="font-size: 85%;">'.__('Explanation:',$this->g_info['ShortName']).'</h4>
			<ul class="mybullet" style="font-size: 85%;">
				<li>
					<strong>'.__('Type "External URL" and "Search String":',$this->g_info['ShortName']).'</strong> '.__('Generally, the class you have entered in the field &raquo;External links:&laquo; is applied to all external links. However, you can enter classes for specific external links, so for example links to Wikipedia, YouTube, Flickr, IMDB, etc. <br />If the domain name of the external link contains the value you have entered in the field &raquo;Search string&laquo;, the according &raquo;CSS class&laquo; will be applied. <br />Examples for \'Search string\': <em>wikipedia, flickr.com, imdb.com</em>. Examples for \'CSS class\': <em>liwikipedia, liflickr, liimdb</em><br />Also, if you use for example <em>wikipedia</em> instead of <em>en.wikipedia.org</em>, it will also work for links like <em>http://www.wikipedia.org/...</em> or <em>http://wikipedia.de/...</em>.<br />You can enter multiple values in the field &raquo;Search string&laquo;; please use pipe (\'|\') or comma (\',\') as separator.',$this->g_info['ShortName']).'
				</li>
				<li>
					<strong>'.__('Type "File Extension":',$this->g_info['ShortName']).'</strong> '.__('In addition to the indication of external/internal links or the type of external link, you can indicate different filename extensions. Just select "File Extension" in the drop-down box, enter the approriate filename extension in the field &raquo;Search string&laquo;, e.g. pdf, xls, txt, png, etc. Like for external URLs, enter the appropriate CSS class in the field &raquo;CSS class&laquo;. <br />Please note that filename extensions do have the highest priority, that means if you have defined a filename extension for "jpg" as well as an external URL "flickr.com", the css class for the file extension "jpg" is being used (and not the css class for flickr) if you have a link like <em>http://www.flickr.com/folder17/summer.jpg</em>.',$this->g_info['ShortName']).'
				</li>
				<li>
					<strong>'.__('Adding more rules:',$this->g_info['ShortName']).'</strong> ' . __('Just fill out the fields and save the options, then 5 more empty fields will appear. In this way you can add an unlimited number of rules.',$this->g_info['ShortName']).'
				</li>
			</ul>
			</div>
			'

		);

		//Add options
		$this->AddContentMain(__('Expert options',$this->g_info['ShortName']), 
			'<h4>'.__('Blog URLs (your internal URLs)',$this->g_info['ShortName']).'</h4>'.
			'<p>' . __('Enter one ore more blog URLs. Separate multiple URLs with line breaks. Links that are based on these URLs are considered as internal links. Chars which follow the',$this->g_info['ShortName']) 
			.  ' <a href="http://en.wikipedia.org/wiki/Top-level_domain">' 
			. __('tld',$this->g_info['ShortName'])
			. '</a> '
			. __('(e.g. <em>xxx</em> in <em>http://www.test.google.com/xxx</em>) will be removed.<br />Your current blog URL: ',$this->g_info['ShortName'])
			. '<em>' . get_bloginfo('url') . '</em>'
			. '<br />' . __('Normally you don\'t need to change anything here and the blog URL is just fine.',$this->g_info['ShortName'])
			. '<textarea name="mwli_blogurls" id="mwli_blogurls" cols="100%" rows="2" >' . $this->COPTHTML('mwli_blogurls') . '</textarea>'
			. '<h4>'.__('Apply CSS Class Attributes also to images',$this->g_info['ShortName']).'</h4>'
			. '<ul style="list-style: none; padding-top:10px;">
				<li>
					<input name="mwli_toimages" type="checkbox" id="mwli_toimages" value="1" '.($this->COPTHTML('mwli_toimages')=='1'?'checked="checked"':'').' /> 
					<label for="mwli_toimages">'.__('Apply class attributes also to images (<em>&lt;img&gt;</em> tags inside the <em>&lt;a&gt;</em> tags)',$this->g_info['ShortName']).'</label>		
	 				<div style="padding-left: 20px;"><small>'.__('Normally it does not make sense to activate this option.',$this->g_info['ShortName']).'</small></div>
				</li>
			</ul>'
			);



		// Sidebar, we can also add individual items...
		$this->PrepareStandardSidebar();
		
		$this->GetGeneratedOptionsPage();

	}


} // class

if( !isset($myLinkIndication)  ) {
	// Create a new instance of your plugin that utilizes the WordpressPluginFramework and initialize the instance.
	$myLinkIndication = new LinkIndication();

	$myLinkIndication->Initialize( 
		// 1. We define the plugin information now and do not use get_plugin_data() due to performance.
		array(	 
			# Plugin name
				'Name' => 			'Link Indication',
			# Author of the plugin
				'Author' => 		'Michael W&ouml;hrer',
			# Authot URI
				'AuthorURI' => 		'http://sw-guide.de/',
			# Plugin URI
				'PluginURI' => 		'http://sw-guide.de/wordpress/plugins/link-indication/',
			# Support URI: E.g. WP or plugin forum, wordpress.org tags, etc.
				'SupportURI' => 	'http://wordpress.org/tags/link-indication',
			# Name of the options for the options database table
				'OptionName' => 	'plugin_link-indication',
			# Old option names to delete from the options table
				'DeleteOldOpt' =>	array('mw_linkindication_plugin', 'plugin_linkindication', 'plugin_linkindication3'),
			# Plugin version
				'Version' => 		'4.3',
			# First plugin version of which we do not reset the plugin options to default;
			# Normally we reset the plugin's options after an update; but if we for example
			# update the plugin from version 2.3 to 2.4 und did only do minor changes and
			# not any option modifications, we should enter '2.3' here. In this example
			# options are being reset to default only if the old plugin version was < 2.3.
				'UseOldOpt' => 		'4.0',
			# Copyright year(s)
				'CopyrightYear' => 	'2006-2010',
			# Minimum WordPress version
				'MinWP' => 			'2.3',				
			# Do not change; full path and filename of the plugin
				'PluginFile' => 	__FILE__,
			# Used for language file, nonce field security, etc.				
				'ShortName' =>		'link-indication',
			),
		// 2. We define the plugin option names and the initial options
		array(
			'pluginversion' => '',	//Please leave it always empty, we fill it later anyway.
			'mwli_blogurls' => get_bloginfo('url'),
			'mwli_applycssclassattr' => '1',
			'mwli_external' => 'liexternal',
			'mwli_ftp' => 'liftp',
			'mwli_mailto' => 'limailto',
			'mwli_internal' => 'liinternal',
			'mwli_imagelinks' => 'liimagelink',
			'mwli_apply' => '1',
			'mwli_toimages' => '',
			'mwli_targetblank' => '',
			'mwli_searchstrings' => array('wikipedia.org', 'wordpress.org', 'pdf','zip|rar|tar|lzh|gz|ace|arj'),
			'mwli_cssclasses' => array('liwikipedia', 'liwp', 'lipdf','lizip'),
			'mwli_types' => array('External URL', 'External URL', 'File Extension', 'File Extension'),
			'mwli_snapr' => '',
			'mwli_nofollow' => '/wp-admin/ wp-login.php wikipedia.org',
			'mwli_usecssfile' => '',
		)
 		);

	$myLinkIndication->ApplyLinkIndication();

} // if( !isset($myLinkIndication) )


?>